<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

// Check if user has any supplier-related permissions
$supplierPermissions = [
    'view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier',
    'search_suppliers', 'process_supplier_payment', 'add_supplier_item'
];

if (!hasAnyPermission($supplierPermissions) && !$isSuperAdmin) {
    $_SESSION['error'] = "You don't have permission to access supplier orders.";
    header('Location: ../pages/dashboard.php');
    exit;
}

$activePage = 'suppliers';
$success = '';
$error = '';

// Handle order submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_order'])) {
    try {
        $pdo->beginTransaction();
        
        $supplier_id = intval($_POST['supplier_id']);
        $product_ids = $_POST['product_id'];
        $quantities = $_POST['quantity'];
        $notes = $_POST['notes'] ?? [];
        $order_date = !empty($_POST['order_date']) ? $_POST['order_date'] : date('Y-m-d');
        $expected_delivery = !empty($_POST['expected_delivery']) ? $_POST['expected_delivery'] : null;
        $order_notes = trim($_POST['order_notes']);
        
        // Validate required fields
        if (empty($supplier_id)) {
            throw new Exception('Please select a supplier.');
        }
        
        if (empty($order_date)) {
            throw new Exception('Order date is required.');
        }
        
        // Validate date format
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $order_date)) {
            throw new Exception('Invalid order date format.');
        }
        
        if ($expected_delivery && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $expected_delivery)) {
            throw new Exception('Invalid expected delivery date format.');
        }
        
        // Validate supplier
        $stmt = $pdo->prepare("SELECT id, name FROM suppliers WHERE id = ?");
        $stmt->execute([$supplier_id]);
        $supplier = $stmt->fetch();
        if (!$supplier) {
            throw new Exception('Invalid supplier selected.');
        }
        
        // Validate that we have at least one item
        if (empty($product_ids) || empty($quantities)) {
            throw new Exception('Please add at least one item to the order.');
        }
        
        // Count valid items (with quantity > 0)
        $valid_items = 0;
        for ($i = 0; $i < count($product_ids); $i++) {
            if (!empty($product_ids[$i]) && intval($quantities[$i]) > 0) {
                $valid_items++;
            }
        }
        
        if ($valid_items === 0) {
            throw new Exception('Please add at least one item with quantity greater than 0.');
        }
        
        // Create the order header
        $stmt = $pdo->prepare("
            INSERT INTO supplier_orders (
                supplier_id, order_date, expected_delivery_date, notes, 
                status, created_by, created_at
            ) VALUES (?, ?, ?, ?, 'pending', ?, NOW())
        ");
        $stmt->execute([
            $supplier_id,
            $order_date,
            $expected_delivery,
            $order_notes,
            $_SESSION['user_id']
        ]);
        $order_id = $pdo->lastInsertId();
        
        // Process each item
        $items_added = 0;
        for ($i = 0; $i < count($product_ids); $i++) {
            $product_id = intval($product_ids[$i]);
            $quantity = intval($quantities[$i]);
            $item_notes = trim($notes[$i] ?? '');
            
            // Skip items with no product selected or zero quantity
            if ($product_id > 0 && $quantity > 0) {
                // Verify product exists
                $stmt = $pdo->prepare("SELECT id FROM products WHERE id = ?");
                $stmt->execute([$product_id]);
                if ($stmt->fetch()) {
                    // Insert order item
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_order_items (
                            order_id, product_id, quantity, notes
                        ) VALUES (?, ?, ?, ?)
                    ");
                    $stmt->execute([$order_id, $product_id, $quantity, $item_notes]);
                    $items_added++;
                }
            }
        }
        
        // Ensure at least one item was actually added
        if ($items_added === 0) {
            throw new Exception('No valid items were added to the order. Please check your selections.');
        }
        
        // Log the action
        logSystemAction(
            'supplier_order_created',
            "Created supplier order #{$order_id} for {$supplier['name']}"
        );
        
        $pdo->commit();
        $success = "Order created successfully! Order ID: #{$order_id} with {$items_added} item(s) for {$supplier['name']}.";
        
        // Redirect to avoid form resubmission
        header("Location: supplier_order.php?success=" . urlencode($success));
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error creating order: " . $e->getMessage();
    }
}

// Handle order update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_order'])) {
    try {
        $pdo->beginTransaction();
        
        $order_id = intval($_POST['order_id']);
        $supplier_id = intval($_POST['supplier_id']);
        $product_ids = $_POST['product_id'];
        $quantities = $_POST['quantity'];
        $notes = $_POST['notes'] ?? [];
        $order_date = !empty($_POST['order_date']) ? $_POST['order_date'] : date('Y-m-d');
        $expected_delivery = !empty($_POST['expected_delivery']) ? $_POST['expected_delivery'] : null;
        $order_notes = trim($_POST['order_notes']);
        
        // Validate required fields
        if (empty($order_id)) {
            throw new Exception('Order ID is required.');
        }
        
        if (empty($supplier_id)) {
            throw new Exception('Please select a supplier.');
        }
        
        if (empty($order_date)) {
            throw new Exception('Order date is required.');
        }
        
        // Validate date format
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $order_date)) {
            throw new Exception('Invalid order date format.');
        }
        
        if ($expected_delivery && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $expected_delivery)) {
            throw new Exception('Invalid expected delivery date format.');
        }
        
        // Validate that order exists and is still in pending status
        $stmt = $pdo->prepare("SELECT id, status FROM supplier_orders WHERE id = ?");
        $stmt->execute([$order_id]);
        $order = $stmt->fetch();
        if (!$order) {
            throw new Exception('Order not found.');
        }
        
        if ($order['status'] !== 'pending') {
            throw new Exception('Only pending orders can be edited.');
        }
        
        // Validate supplier
        $stmt = $pdo->prepare("SELECT id, name FROM suppliers WHERE id = ?");
        $stmt->execute([$supplier_id]);
        $supplier = $stmt->fetch();
        if (!$supplier) {
            throw new Exception('Invalid supplier selected.');
        }
        
        // Validate that we have at least one item
        if (empty($product_ids) || empty($quantities)) {
            throw new Exception('Please add at least one item to the order.');
        }
        
        // Count valid items (with quantity > 0)
        $valid_items = 0;
        for ($i = 0; $i < count($product_ids); $i++) {
            if (!empty($product_ids[$i]) && intval($quantities[$i]) > 0) {
                $valid_items++;
            }
        }
        
        if ($valid_items === 0) {
            throw new Exception('Please add at least one item with quantity greater than 0.');
        }
        
        // Update the order header
        $stmt = $pdo->prepare("
            UPDATE supplier_orders 
            SET supplier_id = ?, order_date = ?, expected_delivery_date = ?, notes = ?, 
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $supplier_id,
            $order_date,
            $expected_delivery,
            $order_notes,
            $order_id
        ]);
        
        // Delete existing order items
        $stmt = $pdo->prepare("DELETE FROM supplier_order_items WHERE order_id = ?");
        $stmt->execute([$order_id]);
        
        // Process each item
        $items_added = 0;
        for ($i = 0; $i < count($product_ids); $i++) {
            $product_id = intval($product_ids[$i]);
            $quantity = intval($quantities[$i]);
            $item_notes = trim($notes[$i] ?? '');
            
            // Skip items with no product selected or zero quantity
            if ($product_id > 0 && $quantity > 0) {
                // Verify product exists
                $stmt = $pdo->prepare("SELECT id FROM products WHERE id = ?");
                $stmt->execute([$product_id]);
                if ($stmt->fetch()) {
                    // Insert order item
                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_order_items (
                            order_id, product_id, quantity, notes
                        ) VALUES (?, ?, ?, ?)
                    ");
                    $stmt->execute([$order_id, $product_id, $quantity, $item_notes]);
                    $items_added++;
                }
            }
        }
        
        // Ensure at least one item was actually added
        if ($items_added === 0) {
            throw new Exception('No valid items were added to the order. Please check your selections.');
        }
        
        // Log the action
        logSystemAction(
            'supplier_order_updated',
            "Updated supplier order #{$order_id} for {$supplier['name']}"
        );
        
        $pdo->commit();
        $success = "Order updated successfully! Order ID: #{$order_id} with {$items_added} item(s) for {$supplier['name']}.";
        
        // Redirect to avoid form resubmission
        header("Location: supplier_order.php?success=" . urlencode($success));
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error updating order: " . $e->getMessage();
    }
}

// Get all suppliers
$stmt = $pdo->query("SELECT id, name, contact_person, phone FROM suppliers ORDER BY name");
$suppliers = $stmt->fetchAll();

// Get all active products
$stmt = $pdo->query("
    SELECT p.id, p.name, p.quantity as current_stock, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE p.status = 'active' 
    ORDER BY p.name
");
$products = $stmt->fetchAll();

// Get recent orders
$stmt = $pdo->prepare("
    SELECT 
        so.id,
        so.order_date,
        so.expected_delivery_date,
        so.status,
        so.notes,
        so.created_at,
        s.name as supplier_name,
        u.username as created_by_username,
        COUNT(soi.id) as item_count
    FROM supplier_orders so
    JOIN suppliers s ON so.supplier_id = s.id
    JOIN users u ON so.created_by = u.id
    LEFT JOIN supplier_order_items soi ON so.id = soi.order_id
    GROUP BY so.id
    ORDER BY so.created_at DESC
    LIMIT 20
");
$stmt->execute();
$recent_orders = $stmt->fetchAll();

// Get store information (with fallback)
$storeSettings = [];
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key = 'store_name'");
    $stmt->execute();
    $result = $stmt->fetch();
    if ($result) {
        $storeSettings['store_name'] = $result['setting_value'];
    }
} catch (Exception $e) {
    // Fallback if store_settings table doesn't exist or has issues
    $storeSettings['store_name'] = 'POS System';
}

// Get SMS and Email settings
$settings = [];
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('enable_sms', 'enable_email')");
    $stmt->execute();
    while ($row = $stmt->fetch()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
} catch (Exception $e) {
    // Fallback if settings can't be loaded
    $settings['enable_sms'] = '0';
    $settings['enable_email'] = '0';
}

// Check for success message in URL
if (isset($_GET['success'])) {
    $success = $_GET['success'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Supplier Orders - <?php echo $storeSettings['store_name'] ?? 'POS System'; ?></title>
    <?php 
    try {
        echo getFaviconLink($pdo); 
    } catch (Exception $e) {
        echo '<link rel="icon" type="image/x-icon" href="../assets/images/favicon.ico">';
    }
    ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <style>
        .order-item-row {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            margin-bottom: 10px;
            padding: 15px;
            background-color: #f8f9fa;
        }
        
        .product-select {
            margin-bottom: 10px;
        }
        
        .status-badge {
            padding: 5px 10px;
            border-radius: 15px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .status-pending { background-color: #fff3cd; color: #856404; }
        .status-approved { background-color: #d1ecf1; color: #0c5460; }
        .status-completed { background-color: #d4edda; color: #155724; }
        .status-cancelled { background-color: #f8d7da; color: #721c24; }
        
        /* Floating Action Button */
        .fab {
            position: fixed;
            bottom: 30px;
            right: 30px;
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: linear-gradient(45deg, #007bff, #0056b3);
            color: white;
            border: none;
            box-shadow: 0 4px 12px rgba(0, 123, 255, 0.4);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            transition: all 0.3s ease;
            z-index: 1050;
        }
        
        .fab:hover {
            transform: scale(1.1);
            box-shadow: 0 6px 20px rgba(0, 123, 255, 0.6);
            color: white;
        }
        
        .fab:focus {
            outline: none;
            box-shadow: 0 6px 20px rgba(0, 123, 255, 0.6);
        }
        
        /* Header button styling */
        .header-create-btn {
            background: linear-gradient(45deg, #28a745, #20c997);
            border: none;
            padding: 12px 24px;
            font-weight: 600;
            box-shadow: 0 2px 8px rgba(40, 167, 69, 0.3);
            transition: all 0.3s ease;
        }
        
        .header-create-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(40, 167, 69, 0.4);
            background: linear-gradient(45deg, #20c997, #28a745);
        }

        /* Ensure modal buttons are visible */
        .modal-footer .btn {
            border: 1px solid transparent !important;
        }
        
        .modal-footer .btn-primary {
            background-color: #007bff !important;
            border-color: #007bff !important;
            color: #fff !important;
        }
        
        .modal-footer .btn-primary:hover {
            background-color: #0056b3 !important;
            border-color: #0056b3 !important;
        }
        
        .modal-footer .btn-secondary {
            background-color: #6c757d !important;
            border-color: #6c757d !important;
            color: #fff !important;
        }
        
        .modal-footer .btn-secondary:hover {
            background-color: #545b62 !important;
            border-color: #545b62 !important;
        }
        
        .modal-footer .btn-outline-info {
            color: #17a2b8 !important;
            border-color: #17a2b8 !important;
            background-color: transparent !important;
        }
        
        .modal-footer .btn-outline-info:hover {
            color: #fff !important;
            background-color: #17a2b8 !important;
        }

        .modal-footer .btn-outline-success {
            color: #28a745 !important;
            border-color: #28a745 !important;
            background-color: transparent !important;
        }
        
        .modal-footer .btn-outline-success:hover {
            color: #fff !important;
            background-color: #28a745 !important;
        }

        .modal-footer .btn-outline-danger {
            color: #dc3545 !important;
            border-color: #dc3545 !important;
            background-color: transparent !important;
        }
        
        .modal-footer .btn-outline-danger:hover {
            color: #fff !important;
            background-color: #dc3545 !important;
        }

        .modal-footer .btn-outline-primary {
            color: #007bff !important;
            border-color: #007bff !important;
            background-color: transparent !important;
        }
        
        .modal-footer .btn-outline-primary:hover {
            color: #fff !important;
            background-color: #007bff !important;
        }
        
        /* Ensure form submit buttons are visible */
        form .btn-primary {
            background-color: #007bff !important;
            border-color: #007bff !important;
            color: #fff !important;
        }
        
        form .btn-primary:hover {
            background-color: #0056b3 !important;
            border-color: #0056b3 !important;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2>Supplier Orders</h2>
                        <p class="text-muted mb-0">Manage and track supplier orders</p>
                    </div>
                    <div>
                        <a href="suppliers.php" class="btn btn-secondary me-2">
                            <i class="fas fa-arrow-left me-2"></i>Back to Suppliers
                        </a>
                        <button class="btn btn-primary header-create-btn" data-bs-toggle="modal" data-bs-target="#orderModal">
                            <i class="fas fa-plus me-2"></i>Create New Order
                        </button>
                    </div>
                </div>



                <?php if ($success): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Recent Orders -->
                <div class="card">
                    <div class="card-header">
                        <div class="row align-items-center">
                            <div class="col-md-4">
                                <h5 class="mb-0">Recent Orders</h5>
                            </div>
                            <div class="col-md-8">
                                <div class="d-flex justify-content-end align-items-center gap-3">
                                    <div class="input-group" style="width: 300px;">
                                        <span class="input-group-text"><i class="fas fa-search"></i></span>
                                        <input type="text" id="orderSearch" class="form-control" placeholder="Search orders...">
                                    </div>
                                    <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#orderModal">
                                        <i class="fas fa-plus me-1"></i>Add Order
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if (empty($recent_orders)): ?>
                            <div class="text-center py-5">
                                <i class="fas fa-clipboard-list fa-4x text-muted mb-4"></i>
                                <h5 class="text-muted mb-3">No orders found</h5>
                                <p class="text-muted mb-4">Create your first supplier order to get started!</p>
                                <button class="btn btn-primary btn-lg" data-bs-toggle="modal" data-bs-target="#orderModal">
                                    <i class="fas fa-plus me-2"></i>Create First Order
                                </button>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Order ID</th>
                                            <th>Supplier</th>
                                            <th>Order Date</th>
                                            <th>Expected Delivery</th>
                                            <th>Items</th>
                                            <th>Status</th>
                                            <th>Created By</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_orders as $order): ?>
                                            <tr>
                                                <td><strong>#<?php echo $order['id']; ?></strong></td>
                                                <td><?php echo htmlspecialchars($order['supplier_name']); ?></td>
                                                <td><?php echo date('Y-m-d', strtotime($order['order_date'])); ?></td>
                                                <td><?php echo $order['expected_delivery_date'] ? date('Y-m-d', strtotime($order['expected_delivery_date'])) : '-'; ?></td>
                                                <td><?php echo $order['item_count']; ?> items</td>
                                                <td>
                                                    <span class="status-badge status-<?php echo $order['status']; ?>">
                                                        <?php echo ucfirst($order['status']); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo htmlspecialchars($order['created_by_username']); ?></td>
                                                <td>
                                                    <div class="btn-group" role="group">
                                                        <?php if ($order['status'] === 'pending'): ?>
                                                        <button class="btn btn-sm btn-outline-warning" 
                                                                onclick="editOrder(<?php echo $order['id']; ?>)"
                                                                data-bs-toggle="tooltip" 
                                                                title="Edit order details">
                                                            <i class="fas fa-edit"></i> Edit
                                                        </button>
                                                        <?php endif; ?>
                                                        <button class="btn btn-sm btn-outline-info" 
                                                                onclick="viewOrderDetails(<?php echo $order['id']; ?>)"
                                                                data-bs-toggle="tooltip" 
                                                                title="View order details and manage status">
                                                            <i class="fas fa-eye"></i> Details
                                                        </button>
                                                        <button class="btn btn-sm btn-outline-secondary" 
                                                                onclick="printOrder(<?php echo $order['id']; ?>)"
                                                                data-bs-toggle="tooltip"
                                                                title="Print order document">
                                                            <i class="fas fa-print"></i> Print
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Order Modal -->
    <div class="modal fade" id="orderModal" tabindex="-1" aria-labelledby="orderModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="orderModalLabel">Create New Order</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" class="needs-validation" novalidate>
                    <div class="modal-body">
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="supplier_id" class="form-label">Supplier *</label>
                                <select class="form-select" id="supplier_id" name="supplier_id" required>
                                    <option value="">Select Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?php echo $supplier['id']; ?>">
                                            <?php echo htmlspecialchars($supplier['name']); ?>
                                            <?php if ($supplier['contact_person']): ?>
                                                - <?php echo htmlspecialchars($supplier['contact_person']); ?>
                                            <?php endif; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <div class="invalid-feedback">Please select a supplier.</div>
                            </div>
                            <div class="col-md-3">
                                <label for="order_date" class="form-label">Order Date *</label>
                                <input type="date" class="form-control" id="order_date" name="order_date" 
                                       value="<?php echo date('Y-m-d'); ?>" required>
                                <div class="invalid-feedback">Please enter order date.</div>
                            </div>
                            <div class="col-md-3">
                                <label for="expected_delivery" class="form-label">Expected Delivery</label>
                                <input type="date" class="form-control" id="expected_delivery" name="expected_delivery">
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="order_notes" class="form-label">Order Notes</label>
                            <textarea class="form-control" id="order_notes" name="order_notes" rows="2" 
                                      placeholder="Optional notes for this order"></textarea>
                        </div>

                        <hr>

                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6>Order Items</h6>
                            <button type="button" class="btn btn-sm btn-success" onclick="addOrderItem()">
                                <i class="fas fa-plus"></i> Add Item
                            </button>
                        </div>

                        <div id="orderItems">
                            <!-- Order items will be added here dynamically -->
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="create_order" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Create Order
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Order Details Modal -->
    <div class="modal fade" id="orderDetailsModal" tabindex="-1" aria-labelledby="orderDetailsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="orderDetailsModalLabel">
                        Order Details - <span id="detailOrderId">#0</span>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <div class="card h-100">
                                <div class="card-header">
                                    <h6 class="mb-0">Order Information</h6>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Supplier:</strong></div>
                                        <div class="col-sm-8" id="detailSupplier">-</div>
                                    </div>
                                    <hr class="my-2">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Order Date:</strong></div>
                                        <div class="col-sm-8" id="detailOrderDate">-</div>
                                    </div>
                                    <hr class="my-2">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Expected Delivery:</strong></div>
                                        <div class="col-sm-8" id="detailExpectedDelivery">-</div>
                                    </div>
                                    <hr class="my-2">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Status:</strong></div>
                                        <div class="col-sm-8">
                                            <span id="detailStatus" class="status-badge">-</span>
                                        </div>
                                    </div>
                                    <hr class="my-2">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Total Items:</strong></div>
                                        <div class="col-sm-8" id="detailTotalItems">0</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card h-100">
                                <div class="card-header">
                                    <h6 class="mb-0">Creation Details</h6>
                                </div>
                                <div class="card-body">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Created By:</strong></div>
                                        <div class="col-sm-8" id="detailCreatedBy">-</div>
                                    </div>
                                    <hr class="my-2">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Created At:</strong></div>
                                        <div class="col-sm-8" id="detailCreatedAt">-</div>
                                    </div>
                                    <hr class="my-2">
                                    <div class="row">
                                        <div class="col-sm-4"><strong>Notes:</strong></div>
                                        <div class="col-sm-8" id="detailNotes">-</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="card">
                        <div class="card-header">
                            <h6 class="mb-0">Order Items</h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-striped">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Ordered Qty</th>
                                            <th>Received Qty</th>
                                            <th>Status</th>
                                            <th>Notes</th>
                                        </tr>
                                    </thead>
                                    <tbody id="orderItemsTableBody">
                                        <!-- Items will be populated via JavaScript -->
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <div class="me-auto">
                        <div class="btn-group" id="statusButtonGroup">
                            <button type="button" id="approveBtn" class="btn btn-sm btn-outline-success" onclick="sendOrderInfoAndApprove()"
                                    title="Approve this order for processing">
                                <i class="fas fa-check me-1"></i>Approve
                            </button>
                            <button type="button" id="cancelBtn" class="btn btn-sm btn-outline-danger" onclick="updateOrderStatus('cancelled')"
                                    title="Cancel this order - cannot be undone">
                                <i class="fas fa-times me-1"></i>Cancel
                            </button>
                        </div>
                        <button type="button" id="resendMessageBtn" class="btn btn-sm btn-outline-primary" onclick="resendOrderMessage()" style="display: none;"
                                title="Resend order message to supplier">
                            <i class="fas fa-paper-plane me-1"></i>Resend Message
                        </button>
                    </div>
                    <div class="d-flex align-items-center gap-2">
                        <div class="form-check me-2">
                            <input class="form-check-input" type="checkbox" id="sendSmsCheckbox" name="send_sms" value="1" 
                                <?php echo ($settings['enable_sms'] ?? '0') === '1' ? 'checked' : ''; ?> 
                                <?php echo ($settings['enable_sms'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                            <label class="form-check-label" for="sendSmsCheckbox">SMS</label>
                        </div>
                        <div class="form-check me-2">
                            <input class="form-check-input" type="checkbox" id="sendWhatsappCheckbox" name="send_whatsapp" value="1" checked>
                            <label class="form-check-label" for="sendWhatsappCheckbox">WhatsApp</label>
                        </div>
                        <div class="form-check me-2">
                            <input class="form-check-input" type="checkbox" id="sendEmailCheckbox" name="send_email" value="1" 
                                <?php echo ($settings['enable_email'] ?? '0') === '1' ? 'checked' : ''; ?> 
                                <?php echo ($settings['enable_email'] ?? '0') !== '1' ? 'disabled' : ''; ?>>
                            <label class="form-check-label" for="sendEmailCheckbox">Email</label>
                        </div>
                        <button type="button" class="btn btn-outline-info btn-sm" onclick="previewOrderMessage()">
                            <i class="fas fa-eye me-1"></i>Preview
                        </button>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="button" class="btn btn-primary" onclick="printOrderDetails()">
                            <i class="fas fa-print me-2"></i>Print Order
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Preview Order Message Modal -->
    <div class="modal fade" id="orderMessagePreviewModal" tabindex="-1" aria-labelledby="orderMessagePreviewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="orderMessagePreviewModalLabel">Order Message Preview</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <pre id="orderMessagePreviewContent" style="white-space: pre-wrap;"></pre>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Floating Action Button -->
    <button class="fab" data-bs-toggle="modal" data-bs-target="#orderModal" title="Create New Order">
        <i class="fas fa-plus"></i>
    </button>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let itemCounter = 0;
        const products = <?php echo json_encode($products); ?>;

        function addOrderItem() {
            const container = document.getElementById('orderItems');
            const itemHtml = `
                <div class="order-item-row" id="item-${itemCounter}">
                    <div class="row">
                        <div class="col-md-5">
                            <label class="form-label">Product *</label>
                            <select class="form-select product-select" name="product_id[]" required onchange="updateStock(this, ${itemCounter})">
                                <option value="">Select Product</option>
                                ${products.map(product => 
                                    `<option value="${product.id}" data-stock="${product.current_stock}">
                                        ${product.name} (Stock: ${product.current_stock})
                                    </option>`
                                ).join('')}
                            </select>
                            <div class="invalid-feedback">Please select a product.</div>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Quantity *</label>
                            <input type="number" class="form-control" name="quantity[]" min="1" required>
                            <div class="invalid-feedback">Please enter quantity.</div>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Notes</label>
                            <input type="text" class="form-control" name="notes[]" placeholder="Optional notes">
                        </div>
                        <div class="col-md-1">
                            <label class="form-label">&nbsp;</label>
                            <button type="button" class="btn btn-danger btn-sm d-block" onclick="removeOrderItem(${itemCounter})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                    <div class="row mt-2">
                        <div class="col-md-12">
                            <small class="text-muted" id="stock-info-${itemCounter}">Select a product to see current stock</small>
                        </div>
                    </div>
                </div>
            `;
            container.insertAdjacentHTML('beforeend', itemHtml);
            itemCounter++;
        }

        function removeOrderItem(id) {
            const item = document.getElementById(`item-${id}`);
            if (item) {
                item.remove();
            }
        }

        function updateStock(select, itemId) {
            const stockInfo = document.getElementById(`stock-info-${itemId}`);
            const selectedOption = select.options[select.selectedIndex];
            
            if (selectedOption.value) {
                const stock = selectedOption.getAttribute('data-stock');
                const productName = selectedOption.text.split(' (Stock:')[0];
                stockInfo.innerHTML = `Current stock for <strong>${productName}</strong>: ${stock} units`;
                
                if (parseInt(stock) === 0) {
                    stockInfo.innerHTML += ' <span class="text-danger">(Out of stock)</span>';
                } else if (parseInt(stock) < 10) {
                    stockInfo.innerHTML += ' <span class="text-warning">(Low stock)</span>';
                }
            } else {
                stockInfo.innerHTML = 'Select a product to see current stock';
            }
        }

        function viewOrderDetails(orderId) {
            console.log('Loading order details for order ID:', orderId);
            // Load order details via AJAX
            fetch(`get_order_details.php?order_id=${orderId}`)
                .then(response => {
                    console.log('Response status:', response.status);
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    console.log('Response data:', data);
                    if (data.success) {
                        displayOrderDetails(data.order, data.items);
                    } else {
                        console.error('API Error:', data);
                        alert('Error loading order details: ' + (data.message || 'Unknown error'));
                    }
                })
                .catch(error => {
                    console.error('Fetch Error:', error);
                    alert('Error loading order details: ' + error.message + '. Please check the browser console for more details.');
                });
        }

        function displayOrderDetails(order, items) {
            const modal = document.getElementById('orderDetailsModal');
            
            // Populate order header
            document.getElementById('detailOrderId').textContent = '#' + order.id;
            document.getElementById('detailSupplier').textContent = order.supplier_name;
            document.getElementById('detailOrderDate').textContent = order.order_date;
            document.getElementById('detailExpectedDelivery').textContent = order.expected_delivery_date || 'Not specified';
            document.getElementById('detailStatus').className = `status-badge status-${order.status}`;
            document.getElementById('detailStatus').textContent = order.status.charAt(0).toUpperCase() + order.status.slice(1);
            document.getElementById('detailCreatedBy').textContent = order.created_by_username;
            document.getElementById('detailCreatedAt').textContent = new Date(order.created_at).toLocaleString();
            document.getElementById('detailNotes').textContent = order.notes || 'No notes';
            
            // Update status buttons visibility
            updateStatusButtons(order.status);
            
            // Populate items table
            const tbody = document.getElementById('orderItemsTableBody');
            tbody.innerHTML = '';
            
            let totalItems = 0;
            items.forEach(item => {
                totalItems += parseInt(item.quantity);
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${item.product_name}</td>
                    <td>${item.quantity}</td>
                    <td>${item.received_quantity || 0}</td>
                    <td>
                        <span class="badge bg-${item.status === 'received' ? 'success' : (item.status === 'partial' ? 'warning' : 'secondary')}">
                            ${item.status.charAt(0).toUpperCase() + item.status.slice(1)}
                        </span>
                    </td>
                    <td>${item.notes || '-'}</td>
                `;
                tbody.appendChild(row);
            });
            
            document.getElementById('detailTotalItems').textContent = totalItems;
            
            // Show modal
            const orderDetailsModal = new bootstrap.Modal(modal);
            orderDetailsModal.show();

            // Set supplier contact info for message sending
            window.currentOrderSupplierPhone = order.supplier_phone || '';
            window.currentOrderSupplierEmail = order.supplier_email || '';
        }

        function printOrderDetails() {
            const orderId = document.getElementById('detailOrderId').textContent.replace('#', '');
            console.log('Printing order details for order:', orderId);
            const printWindow = window.open(`print_supplier_order.php?order_id=${orderId}`, '_blank');
            if (!printWindow) {
                alert('Pop-up blocked! Please allow pop-ups for this site to print orders.');
                return;
            }
            printWindow.focus();
        }

        function updateOrderStatus(newStatus) {
            const orderId = document.getElementById('detailOrderId').textContent.replace('#', '');
            const currentStatus = document.getElementById('detailStatus').textContent.toLowerCase();
            
            if (currentStatus === newStatus) {
                alert(`Order is already ${newStatus}.`);
                return;
            }
            
            if (confirm(`Are you sure you want to ${newStatus} this order?`)) {
                fetch('update_order_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        order_id: orderId,
                        status: newStatus
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Update the status badge in the modal
                        const statusBadge = document.getElementById('detailStatus');
                        statusBadge.className = `status-badge status-${newStatus}`;
                        statusBadge.textContent = newStatus.charAt(0).toUpperCase() + newStatus.slice(1);
                        
                        // Update status buttons visibility
                        updateStatusButtons(newStatus);
                        
                        // Show success message
                        alert('Order status updated successfully!');
                        
                        // Refresh the page to show updated status in the table
                        setTimeout(() => {
                            window.location.reload();
                        }, 1000);
                    } else {
                        alert('Error updating order status: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error updating order status. Please try again.');
                });
            }
        }

        function updateStatusButtons(currentStatus) {
            const approveBtn = document.getElementById('approveBtn');
            const cancelBtn = document.getElementById('cancelBtn');
            const resendMessageBtn = document.getElementById('resendMessageBtn');
            const buttonGroup = document.getElementById('statusButtonGroup');
            
            // Reset all buttons to visible
            approveBtn.style.display = 'inline-block';
            cancelBtn.style.display = 'inline-block';
            resendMessageBtn.style.display = 'none'; // Hide resend message by default
            buttonGroup.style.display = 'block'; // Always show button group
            
            // Hide buttons based on current status
            switch(currentStatus) {
                case 'approved':
                    approveBtn.style.display = 'none';
                    cancelBtn.style.display = 'inline-block';
                    resendMessageBtn.style.display = 'inline-block'; // Show resend message for approved orders
                    break;
                case 'completed':
                    approveBtn.style.display = 'none';
                    cancelBtn.style.display = 'none';
                    resendMessageBtn.style.display = 'inline-block'; // Show resend message for completed orders too
                    break;
                case 'cancelled':
                    approveBtn.style.display = 'none';
                    cancelBtn.style.display = 'none';
                    resendMessageBtn.style.display = 'none'; // Hide resend message for cancelled orders
                    break;
                case 'pending':
                    // All buttons remain visible for pending orders (except resend)
                    approveBtn.style.display = 'inline-block';
                    cancelBtn.style.display = 'inline-block';
                    resendMessageBtn.style.display = 'none'; // Hide resend message for pending orders
                    break;
            }
            
            // Only hide button group if all buttons in it are hidden
            const visibleButtonsInGroup = [approveBtn, cancelBtn].filter(btn => btn.style.display !== 'none');
            if (visibleButtonsInGroup.length === 0) {
                buttonGroup.style.display = 'none';
            } else {
                buttonGroup.style.display = 'block';
            }
        }

        function editOrder(orderId) {
            console.log('Loading order details for editing, order ID:', orderId);
            // Load order details for editing
            fetch(`get_order_details.php?order_id=${orderId}`)
                .then(response => {
                    console.log('Edit response status:', response.status);
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    console.log('Edit response data:', data);
                    if (data.success) {
                        populateEditForm(data.order, data.items);
                        const modal = new bootstrap.Modal(document.getElementById('orderModal'));
                        modal.show();
                    } else {
                        console.error('Edit API Error:', data);
                        alert('Error loading order details: ' + (data.message || 'Unknown error'));
                    }
                })
                .catch(error => {
                    console.error('Edit Fetch Error:', error);
                    alert('Error loading order details: ' + error.message + '. Please check the browser console for more details.');
                });
        }

        function populateEditForm(order, items) {
            // Update modal title
            document.getElementById('orderModalLabel').textContent = 'Edit Order #' + order.id;
            
            // Add hidden field for order ID
            const form = document.querySelector('#orderModal form');
            let orderIdField = form.querySelector('input[name="order_id"]');
            if (!orderIdField) {
                orderIdField = document.createElement('input');
                orderIdField.type = 'hidden';
                orderIdField.name = 'order_id';
                form.appendChild(orderIdField);
            }
            orderIdField.value = order.id;
            
            // Populate form fields
            document.getElementById('supplier_id').value = order.supplier_id;
            document.getElementById('order_date').value = order.order_date;
            document.getElementById('expected_delivery').value = order.expected_delivery_date || '';
            document.getElementById('order_notes').value = order.notes || '';
            
            // Clear existing items
            const container = document.getElementById('orderItems');
            container.innerHTML = '';
            itemCounter = 0;
            
            // Add items from the order
            items.forEach(item => {
                addOrderItem();
                const currentItem = container.lastElementChild;
                const productSelect = currentItem.querySelector('select[name="product_id[]"]');
                const quantityInput = currentItem.querySelector('input[name="quantity[]"]');
                const notesInput = currentItem.querySelector('input[name="notes[]"]');
                
                productSelect.value = item.product_id;
                quantityInput.value = item.quantity;
                notesInput.value = item.notes || '';
                
                // Update stock info
                updateStock(productSelect, itemCounter - 1);
            });
            
            // Change form action to update instead of create
            const submitButton = form.querySelector('button[name="create_order"]');
            submitButton.name = 'update_order';
            submitButton.innerHTML = '<i class="fas fa-save me-2"></i>Update Order';
        }

        function previewOrderMessage() {
            // Compose the order message for preview
            const orderId = document.getElementById('detailOrderId').textContent;
            const supplier = document.getElementById('detailSupplier').textContent;
            const orderDate = document.getElementById('detailOrderDate').textContent;
            const status = document.getElementById('detailStatus').textContent;
            const createdBy = document.getElementById('detailCreatedBy').textContent;
            const createdAt = document.getElementById('detailCreatedAt').textContent;
            const notes = document.getElementById('detailNotes').textContent;
            const totalItems = document.getElementById('detailTotalItems').textContent;

            // Get store name from page title
            const storeName = document.title.split(' - ')[1] || 'Our Store';

            // Get order items table
            const itemsTable = document.getElementById('orderItemsTableBody');
            let itemsText = '';
            for (let row of itemsTable.rows) {
                const productName = row.cells[0].textContent.trim();
                const quantity = row.cells[1].textContent.trim();
                const status = row.cells[3].textContent.trim();
                itemsText += `• ${productName} - Quantity: ${quantity} (${status})\n`;
            }

            // Choose greeting based on status
            const isCompleted = status.toLowerCase() === 'completed';
            const greeting = isCompleted 
                ? `The following order was well received!!`
                : `We would like to make the following order:`;

            const message = 
                `Hello, this is ${storeName}. ${greeting}\n\n` +
                `📋 SUPPLIER ORDER DETAILS\n` +
                `━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n` +
                `Order ID: ${orderId}\n` +
                `Supplier: ${supplier}\n` +
                `Order Date: ${orderDate}\n` +
                `Total Items: ${totalItems}\n\n` +
                `📦 ORDER ITEMS:\n` +
                `${itemsText}\n` +
                `━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n` +
                `Created by: ${createdBy}\n` +
                `Date: ${createdAt}\n` +
                `${notes && notes !== 'No notes' ? `\nNotes: ${notes}` : ''}\n\n` +
                `Thank you for your business.\n` +
                `For any queries, please contact us.`;

            document.getElementById('orderMessagePreviewContent').textContent = message;
            const previewModal = new bootstrap.Modal(document.getElementById('orderMessagePreviewModal'));
            previewModal.show();
        }

        function sendOrderInfoAndApprove() {
            // Get selected channels
            const sendSms = document.getElementById('sendSmsCheckbox').checked;
            const sendWhatsapp = document.getElementById('sendWhatsappCheckbox').checked;
            const sendEmail = document.getElementById('sendEmailCheckbox').checked;

            // Compose the order message
            const orderId = document.getElementById('detailOrderId').textContent;
            const supplier = document.getElementById('detailSupplier').textContent;
            const orderDate = document.getElementById('detailOrderDate').textContent;
            const status = document.getElementById('detailStatus').textContent;
            const createdBy = document.getElementById('detailCreatedBy').textContent;
            const createdAt = document.getElementById('detailCreatedAt').textContent;
            const notes = document.getElementById('detailNotes').textContent;
            const totalItems = document.getElementById('detailTotalItems').textContent;
            
            // Get store name from page title
            const storeName = document.title.split(' - ')[1] || 'Our Store';
            
            const itemsTable = document.getElementById('orderItemsTableBody');
            let itemsText = '';
            for (let row of itemsTable.rows) {
                const productName = row.cells[0].textContent.trim();
                const quantity = row.cells[1].textContent.trim();
                const status = row.cells[3].textContent.trim();
                itemsText += `• ${productName} - Quantity: ${quantity} (${status})\n`;
            }

            // Choose greeting based on status
            const isCompleted = status.toLowerCase() === 'completed';
            const greeting = isCompleted 
                ? `The following order was well received!!`
                : `We would like to make the following order:`;

            const message = 
                `Hello, this is ${storeName}. ${greeting}\n\n` +
                `📋 SUPPLIER ORDER DETAILS\n` +
                `━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n` +
                `Order ID: ${orderId}\n` +
                `Supplier: ${supplier}\n` +
                `Order Date: ${orderDate}\n` +
                `Total Items: ${totalItems}\n\n` +
                `📦 ORDER ITEMS:\n` +
                `${itemsText}\n` +
                `━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n` +
                `Created by: ${createdBy}\n` +
                `Date: ${createdAt}\n` +
                `${notes && notes !== 'No notes' ? `\nNotes: ${notes}` : ''}\n\n` +
                `Thank you for your business.\n` +
                `For any queries, please contact us.`;

            // Get supplier contact info from the modal (should be loaded in JS when displaying details)
            // For this example, let's assume you have global variables set when loading order details:
            // window.currentOrderSupplierPhone, window.currentOrderSupplierEmail
            const phone = window.currentOrderSupplierPhone || '';
            const email = window.currentOrderSupplierEmail || '';

            // Send via selected channels
            let sendPromises = [];
            if (sendSms && phone) {
                sendPromises.push(fetch('../api/send_supplier_order_message.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ type: 'sms', phone, message })
                }).then(res => res.json()));
            }
            if (sendEmail && email) {
                sendPromises.push(fetch('../api/send_supplier_order_message.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ type: 'email', email, message, subject: `Supplier Order ${orderId} - ${supplier}` })
                }).then(res => res.json()));
            }
            if (sendWhatsapp && phone) {
                // WhatsApp: open link in new tab
                const whatsappUrl = `https://wa.me/${phone.replace(/[^0-9]/g, '')}?text=${encodeURIComponent(message)}`;
                window.open(whatsappUrl, '_blank');
            }

            // After all sends, approve the order
            Promise.all(sendPromises).then(results => {
                let errors = results.filter(r => !r.success).map(r => r.message || r.error);
                if (errors.length > 0) {
                    alert('Some messages failed to send:\n' + errors.join('\n'));
                } else {
                    alert('Order information sent successfully!');
                }
                // Now approve the order
                updateOrderStatus('approved');
            });
        }

        function resendOrderMessage() {
            // Get selected channels
            const sendSms = document.getElementById('sendSmsCheckbox').checked;
            const sendWhatsapp = document.getElementById('sendWhatsappCheckbox').checked;
            const sendEmail = document.getElementById('sendEmailCheckbox').checked;

            const orderId = document.getElementById('detailOrderId').textContent.replace('#', '');
            const supplier = document.getElementById('detailSupplier').textContent;
            const orderDate = document.getElementById('detailOrderDate').textContent;
            const status = document.getElementById('detailStatus').textContent;
            const createdBy = document.getElementById('detailCreatedBy').textContent;
            const createdAt = document.getElementById('detailCreatedAt').textContent;
            const notes = document.getElementById('detailNotes').textContent;
            const totalItems = document.getElementById('detailTotalItems').textContent;

            // Get store name from page title
            const storeName = document.title.split(' - ')[1] || 'Our Store';

            const itemsTable = document.getElementById('orderItemsTableBody');
            let itemsText = '';
            for (let row of itemsTable.rows) {
                const productName = row.cells[0].textContent.trim();
                const quantity = row.cells[1].textContent.trim();
                const status = row.cells[3].textContent.trim();
                itemsText += `• ${productName} - Quantity: ${quantity} (${status})\n`;
            }

            // Choose greeting based on status
            const isCompleted = status.toLowerCase() === 'completed';
            const greeting = isCompleted 
                ? `The following order was well received!!`
                : `We would like to make the following order:`;

            const message = 
                `Hello, this is ${storeName}. ${greeting}\n\n` +
                `📋 SUPPLIER ORDER DETAILS\n` +
                `━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n` +
                `Order ID: ${orderId}\n` +
                `Supplier: ${supplier}\n` +
                `Order Date: ${orderDate}\n` +
                `Total Items: ${totalItems}\n\n` +
                `📦 ORDER ITEMS:\n` +
                `${itemsText}\n` +
                `━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n` +
                `Created by: ${createdBy}\n` +
                `Date: ${createdAt}\n` +
                `${notes && notes !== 'No notes' ? `\nNotes: ${notes}` : ''}\n\n` +
                `Thank you for your business.\n` +
                `For any queries, please contact us.`;

            // Get supplier contact info from the modal (should be loaded in JS when displaying details)
            const phone = window.currentOrderSupplierPhone || '';
            const email = window.currentOrderSupplierEmail || '';

            let sendPromises = [];
            if (sendSms && phone) {
                sendPromises.push(fetch('../api/send_supplier_order_message.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ type: 'sms', phone, message })
                }).then(res => res.json()));
            }
            if (sendEmail && email) {
                sendPromises.push(fetch('../api/send_supplier_order_message.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ type: 'email', email, message, subject: `Supplier Order ${orderId} - ${supplier}` })
                }).then(res => res.json()));
            }
            if (sendWhatsapp && phone) {
                // WhatsApp: open link in new tab
                const whatsappUrl = `https://wa.me/${phone.replace(/[^0-9]/g, '')}?text=${encodeURIComponent(message)}`;
                window.open(whatsappUrl, '_blank');
            }

            Promise.all(sendPromises).then(results => {
                let errors = results.filter(r => !r.success).map(r => r.message || r.error);
                if (errors.length > 0) {
                    alert('Some messages failed to send:\n' + errors.join('\n'));
                } else {
                    alert('Order message resent successfully!');
                }
            }).catch(error => {
                console.error('Error resending message:', error);
                alert('Error resending order message. Please try again.');
            });
        }

        // Search functionality
        function initializeSearch() {
            const searchInput = document.getElementById('orderSearch');
            if (searchInput) {
                searchInput.addEventListener('input', function() {
                    const searchTerm = this.value.toLowerCase();
                    const rows = document.querySelectorAll('.table tbody tr');
                    
                    rows.forEach(row => {
                        const cells = row.querySelectorAll('td');
                        let rowText = '';
                        
                        // Concatenate text from all cells except the actions column
                        for (let i = 0; i < cells.length - 1; i++) {
                            rowText += cells[i].textContent.toLowerCase() + ' ';
                        }
                        
                        if (rowText.includes(searchTerm)) {
                            row.style.display = '';
                        } else {
                            row.style.display = 'none';
                        }
                    });
                });
            }
        }

        // Initialize with one item
        document.addEventListener('DOMContentLoaded', function() {
            addOrderItem();
            
            // Initialize search functionality
            initializeSearch();
            
            // Initialize tooltips
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            const tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
            
            // Reset form when modal is closed
            const orderModal = document.getElementById('orderModal');
            orderModal.addEventListener('hidden.bs.modal', function() {
                const form = this.querySelector('form');
                form.reset();
                form.classList.remove('was-validated');
                
                // Reset modal title
                document.getElementById('orderModalLabel').textContent = 'Create New Order';
                
                // Remove order ID field if it exists
                const orderIdField = form.querySelector('input[name="order_id"]');
                if (orderIdField) {
                    orderIdField.remove();
                }
                
                // Reset submit button
                const submitButton = form.querySelector('button[name="update_order"], button[name="create_order"]');
                if (submitButton) {
                    submitButton.name = 'create_order';
                    submitButton.innerHTML = '<i class="fas fa-save me-2"></i>Create Order';
                }
                
                // Clear order items and add one empty item
                const container = document.getElementById('orderItems');
                container.innerHTML = '';
                itemCounter = 0;
                addOrderItem();
            });
            
            // Ensure modal has items when shown
            orderModal.addEventListener('shown.bs.modal', function() {
                const container = document.getElementById('orderItems');
                // If no items exist, add one
                if (container.children.length === 0) {
                    console.log('Modal shown with no items, adding one');
                    addOrderItem();
                }
            });
            
            // Form validation
            const forms = document.querySelectorAll('.needs-validation');
            Array.from(forms).forEach(form => {
                form.addEventListener('submit', event => {
                    let isValid = form.checkValidity();
                    
                    // Additional validation: check if at least one item has product and quantity
                    const productSelects = form.querySelectorAll('select[name="product_id[]"]');
                    const quantityInputs = form.querySelectorAll('input[name="quantity[]"]');
                    let hasValidItem = false;
                    
                    for (let i = 0; i < productSelects.length; i++) {
                        if (productSelects[i].value && quantityInputs[i].value && parseInt(quantityInputs[i].value) > 0) {
                            hasValidItem = true;
                            break;
                        }
                    }
                    
                    if (!hasValidItem) {
                        alert('Please add at least one item with a valid product and quantity greater than 0.');
                        isValid = false;
                    }
                    
                    if (!isValid) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    form.classList.add('was-validated');
                });
            });
        });
    </script>
</body>
</html> 